//
//  Notifications.swift
//  Do It
//
//  Created by Jim Dovey on 3/30/20.
//  Copyright © 2020 Jim Dovey. All rights reserved.
//

import UserNotifications

final class NotificationManager {
    private var scheduledItemIDs: Set<UUID> = []
    private let formatter: DateFormatter = {
        let f = DateFormatter()
        f.dateStyle = .medium
        f.timeStyle = .short
        f.doesRelativeDateFormatting = true
        return f
    }()
    
    static let shared = NotificationManager()
    
    private enum Action: String, CaseIterable {
        case complete = "COMPLETE_ACTION"
        case snooze = "SNOOZE_ACTION"
        case show = "SHOW_ACTION"
        
        var title: String {
            switch self {
            case .complete:
                return NSLocalizedString("Accept", comment: "Notification action")
            case .snooze:
                return NSLocalizedString("Snooze", comment: "Notification action")
            case .show:
                return NSLocalizedString("Show", comment: "Notification action")
            }
        }
        
        var options: UNNotificationActionOptions {
            switch self {
            case .complete: return .authenticationRequired
            case .snooze: return []
            case .show: return .foreground
            }
        }
    }
    
    private enum Category: String {
        case alarm = "ALARM_CATEGORY"
        
        var options: UNNotificationCategoryOptions {
            switch self {
            case .alarm:
                #if !os(macOS)
                return [.allowAnnouncement, .customDismissAction]
                #else
                return .customDismissAction
                #endif
            }
        }
    }
    
    init() {
        setupScheduledIDs()
        registerCategories()
    }
    
    func preflightFirstLaunchScheduling(for data: DataCenter) {
        UNUserNotificationCenter.current().getNotificationSettings { settings in
            switch settings.authorizationStatus {
            case .notDetermined:
                self.obtainAuthorization(for: data) {
                    guard $0 else { return }
                    self.scheduleAllNotifications(for: data)
                }
            case .authorized, .provisional:
                self.scheduleAllNotifications(for: data)
            case .denied:
                break
            @unknown default:
                break
            }
        }
    }
    
    private func obtainAuthorization(for data: DataCenter,
                                     completion: @escaping (Bool) -> Void) {
        let options: UNAuthorizationOptions
        #if os(macOS)
        options = [.alert, .sound]
        #else
        options = [.alert, .announcement, .sound]
        #endif
        
        UNUserNotificationCenter.current()
            .requestAuthorization(options: options) { result, error in
                if let err = error {
                    print("Error obtaining notification authorization: \(err)")
                }
                completion(result)
        }
    }
    
    private func scheduleAllNotifications(for data: DataCenter) {
        let center = UNUserNotificationCenter.current()
        center.removeAllPendingNotificationRequests()
        center.removeAllDeliveredNotifications()
        
        let now = Date()
        data.todoItems.lazy
            .filter { !$0.complete && $0.date != nil && $0.date! > now }
            .map { ($0, TimeInterval(0)) }
            .forEach(scheduleNotification(for:timeOffset:))
    }
    
    private func setupScheduledIDs() {
        let group = DispatchGroup()
        group.enter()
        UNUserNotificationCenter.current().getPendingNotificationRequests { requests in
            let uuids = requests.map { $0.identifier }
                .compactMap(UUID.init(uuidString:))
            self.scheduledItemIDs.formUnion(uuids)
            group.leave()
        }
        group.wait()
    }
    
    private func registerCategories() {
        let actions = Action.allCases.map {
            UNNotificationAction(
                identifier: $0.rawValue,
                title: $0.title,
                options: $0.options)
        }
        let alarmCategory = UNNotificationCategory(
            identifier: Category.alarm.rawValue,
            actions: actions,
            intentIdentifiers: [],
            hiddenPreviewsBodyPlaceholder: "",
            options: Category.alarm.options)
        
        UNUserNotificationCenter.current()
            .setNotificationCategories([alarmCategory])
    }
    
    private func _scheduleNotification(for item: TodoItem, at date: Date) {
        UNUserNotificationCenter.current().getNotificationSettings { settings in
            let content = UNMutableNotificationContent()
            content.title = item.title
            content.body = self.formatter.string(from: date)
            content.userInfo = ["ITEM_ID": item.id.uuidString]
            content.categoryIdentifier = Category.alarm.rawValue
            
            if settings.soundSetting == .enabled {
                content.sound = .default
            }
            
            let components = Calendar.current.dateComponents(in: TimeZone.current,
                                                             from: date)
            let trigger = UNCalendarNotificationTrigger(dateMatching: components,
                                                        repeats: false)
            let request = UNNotificationRequest(identifier: item.id.uuidString,
                                                content: content, trigger: trigger)
            
            self.scheduledItemIDs.insert(item.id)
            UNUserNotificationCenter.current().add(request) {
                if let error = $0 {
                    print("Error scheduling notification: \(error)")
                    self.scheduledItemIDs.remove(item.id)
                }
            }
        }
    }
    
    func scheduleNotification(for item: TodoItem, timeOffset: TimeInterval = 0) {
        guard !scheduledItemIDs.contains(item.id) else { return }
        guard let date = item.date else { return }
        
        _scheduleNotification(for: item, at: date)
    }
    
    func cancelNotification(for item: TodoItem) {
        UNUserNotificationCenter.current()
            .removePendingNotificationRequests(withIdentifiers: [item.id.uuidString])
    }
    
    func handleNotificationResponse(_ response: UNNotificationResponse,
                                    _ dataCenter: DataCenter) -> TodoItem? {
        let info = response.notification.request.content.userInfo
        guard
            let itemIDStr = info["ITEM_ID"] as? String,
            let itemID = UUID(uuidString: itemIDStr),
            let action = Action(rawValue: response.actionIdentifier)
            else { return nil }

        UNUserNotificationCenter.current()
            .removeDeliveredNotifications(
                withIdentifiers: [response.notification.request.identifier])
        self.scheduledItemIDs.remove(itemID)
        
        switch action {
        case .complete:
            guard var item = dataCenter.item(withID: itemID) else {
                return nil
            }
            item.completed = Date()
            dataCenter.updateTodoItem(item)
        case .snooze:
            // new notification, ten minutes from now
            guard let item = dataCenter.item(withID: itemID) else {
                return nil
            }
            let date = Date(timeIntervalSinceNow: 600)
            _scheduleNotification(for: item, at: date)
        case .show:
            // hand the item back up
            return dataCenter.item(withID: itemID)
        }
        
        return nil
    }
}
